<?php
header('Content-Type: application/json');
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/functions.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true) ?: [];

// Validate action
$action = $input['action'] ?? '';
if (empty($action)) {
    echo json_encode(['success' => false, 'message' => 'No action specified']);
    exit;
}

try {
    switch ($action) {
        case 'getUserData':
            $userId = $input['userId'] ?? 0;
            if (empty($userId)) {
                throw new Exception('User ID is required');
            }
            
            // Get user data
            $user = getUserData($userId);
            if (!$user) {
                // Create new user if doesn't exist
                $user = createUser($userId);
            }
            
            // Get system settings
            $settings = getSystemSettings();
            
            echo json_encode([
                'success' => true,
                'balance' => $user['balance'],
                'adsWatched' => $user['total_ads_watched'],
                'adsRequired' => $settings['min_ads_to_watch'],
                'rewardPerAd' => $settings['reward_per_ad'],
                'referralCode' => $user['referral_code'],
                'referredCount' => getReferredCount($user['id'])
            ]);
            break;
            
        case 'initAdWatch':
            $userId = $input['userId'] ?? 0;
            if (empty($userId)) {
                throw new Exception('User ID is required');
            }
            
            // Check if user exists
            $user = getUserData($userId);
            if (!$user) {
                throw new Exception('User not found');
            }
            
            // Here you might want to add rate limiting or other checks
            
            echo json_encode(['success' => true]);
            break;
            
        case 'completeAdWatch':
            $userId = $input['userId'] ?? 0;
            if (empty($userId)) {
                throw new Exception('User ID is required');
            }
            
            // Get user and settings
            $user = getUserData($userId);
            $settings = getSystemSettings();
            
            if (!$user) {
                throw new Exception('User not found');
            }
            
            // Update ads watched count
            $newAdsCount = $user['total_ads_watched'] + 1;
            updateUserAdsCount($userId, $newAdsCount);
            
            // Add reward to balance
            $rewardAmount = $settings['reward_per_ad'];
            $newBalance = $user['balance'] + $rewardAmount;
            updateUserBalance($userId, $newBalance);
            
            // Log the ad watch
            logAdWatch($userId, $rewardAmount);
            
            echo json_encode([
                'success' => true,
                'newBalance' => $newBalance,
                'newAdsCount' => $newAdsCount,
                'rewardAmount' => $rewardAmount
            ]);
            break;
            
        case 'requestWithdraw':
            $userId = $input['userId'] ?? 0;
            $amount = floatval($input['amount'] ?? 0);
            $wallet = trim($input['wallet'] ?? '');
            
            if (empty($userId) || empty($amount) || empty($wallet)) {
                throw new Exception('All fields are required');
            }
            
            if ($amount < 10) {
                throw new Exception('Minimum withdrawal is 10 coins');
            }
            
            // Check user balance
            $user = getUserData($userId);
            if (!$user) {
                throw new Exception('User not found');
            }
            
            if ($user['balance'] < $amount) {
                throw new Exception('Insufficient balance');
            }
            
            // Deduct balance
            $newBalance = $user['balance'] - $amount;
            updateUserBalance($userId, $newBalance);
            
            // Create withdrawal request
            createWithdrawalRequest($userId, $amount, $wallet);
            
            echo json_encode([
                'success' => true,
                'message' => 'Withdrawal request submitted'
            ]);
            break;
            
        case 'getMonetagScript':
            $settings = getSystemSettings();
            echo json_encode([
                'success' => true,
                'script' => $settings['monetag_script'] ?? ''
            ]);
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
    }
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

// Database helper functions from includes/functions.php would be here
// These would include implementations for:
// - getUserData()
// - createUser()
// - getSystemSettings()
// - updateUserAdsCount()
// - updateUserBalance()
// - logAdWatch()
// - getReferredCount()
// - createWithdrawalRequest()
?>