<?php
$token = "YOUR_BOT_TOKEN";
$api_url = "https://api.telegram.org/bot$token/";
$smm_api_url = "https://smmgen.com/api/v2"; // SMM panel
$smm_api_key = "YOUR_SMM_API_KEY";

// Files
define('USER_FILE', 'users.json');
define('ORDER_FILE', 'orders.json');
define('SERVICE_FILE', 'services.json');

// Telegram API (fast cURL)
function apiRequest($method, $data) {
    $url = $GLOBALS['api_url'] . $method;
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $data,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_TIMEOUT => 10
    ]);
    $response = curl_exec($ch);
    curl_close($ch);
    return json_decode($response, true);
}

// JSON helpers
function loadJson($file) {
    if (!file_exists($file)) return [];
    return json_decode(file_get_contents($file), true) ?? [];
}

function saveJson($file, $data) {
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
}

// Load service config
function getService($key) {
    $services = loadJson(SERVICE_FILE);
    return $services[$key] ?? null;
}

// External API call to place order
function placeOrder($service_id, $link, $quantity) {
    global $smm_api_key, $smm_api_url;
    $post = [
        'key' => $smm_api_key,
        'action' => 'add',
        'service' => $service_id,
        'link' => $link,
        'quantity' => $quantity
    ];

    $ch = curl_init($smm_api_url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($post),
    ]);
    $response = curl_exec($ch);
    curl_close($ch);
    return json_decode($response, true);
}

// MAIN LOGIC
$update = json_decode(file_get_contents('php://input'), true);
$message = $update['message'] ?? null;
$callback = $update['callback_query'] ?? null;

if ($message) {
    $chat_id = $message['chat']['id'];
    $text = trim($message['text'] ?? '');
    $users = loadJson(USER_FILE);
    $user = &$users[$chat_id];

    if (!isset($user)) $user = ['step' => null, 'order_temp' => []];

    // Step handling
    if ($user['step'] === 'waiting_link') {
        $user['order_temp']['link'] = $text;
        $user['step'] = 'waiting_quantity';
        apiRequest("sendMessage", [
            'chat_id' => $chat_id,
            'text' => "🔢 Please send the quantity:"
        ]);
        saveJson(USER_FILE, $users);
        exit;
    }

    if ($user['step'] === 'waiting_quantity') {
        if (!is_numeric($text)) {
            apiRequest("sendMessage", ['chat_id' => $chat_id, 'text' => "❌ Please send a valid number."]);
            exit;
        }

        $quantity = intval($text);
        $service_key = $user['order_temp']['service_key'];
        $service = getService($service_key);

        if (!$service) {
            apiRequest("sendMessage", ['chat_id' => $chat_id, 'text' => "❌ Service not found."]);
            $user['step'] = null;
            saveJson(USER_FILE, $users);
            exit;
        }

        if ($quantity < $service['min_order']) {
            apiRequest("sendMessage", [
                'chat_id' => $chat_id,
                'text' => "❌ Minimum order is {$service['min_order']}."
            ]);
            exit;
        }

        $price = $quantity * $service['price_per_unit'];

        $user['order_temp']['quantity'] = $quantity;
        $user['order_temp']['price'] = $price;
        $user['step'] = 'waiting_confirm';
        saveJson(USER_FILE, $users);

        $msg = "🛒 Order Summary:\n";
        $msg .= "🔗 Link: {$user['order_temp']['link']}\n";
        $msg .= "📦 Service: {$service['name']}\n";
        $msg .= "🔢 Quantity: $quantity\n";
        $msg .= "💰 Price: $" . number_format($price, 2);

        $buttons = [
            'inline_keyboard' => [
                [['text' => "✅ Confirm", 'callback_data' => "confirm_order"]],
                [['text' => "❌ Cancel", 'callback_data' => "cancel_order"]]
            ]
        ];

        apiRequest("sendMessage", [
            'chat_id' => $chat_id,
            'text' => $msg,
            'reply_markup' => json_encode($buttons)
        ]);
        exit;
    }

    // Start or default
    if ($text == "/start") {
        $services = loadJson(SERVICE_FILE);
        $keyboard = ['inline_keyboard' => []];
        foreach ($services as $key => $svc) {
            $keyboard['inline_keyboard'][] = [
                ['text' => $svc['name'], 'callback_data' => "select_service_$key"]
            ];
        }
        apiRequest("sendMessage", [
            'chat_id' => $chat_id,
            'text' => "🛍️ Available Services:\nSelect a service to continue:",
            'reply_markup' => json_encode($keyboard)
        ]);
    }
}

if ($callback) {
    $chat_id = $callback['from']['id'];
    $data = $callback['data'];
    $users = loadJson(USER_FILE);
    $user = &$users[$chat_id];
    if (!isset($user)) $user = ['step' => null, 'order_temp' => []];

    if (strpos($data, 'select_service_') === 0) {
        $key = str_replace('select_service_', '', $data);
        $service = getService($key);

        if (!$service) {
            apiRequest("answerCallbackQuery", [
                'callback_query_id' => $callback['id'],
                'text' => "Service not found!"
            ]);
            exit;
        }

        $user['step'] = 'waiting_link';
        $user['order_temp'] = ['service_key' => $key];
        saveJson(USER_FILE, $users);

        apiRequest("sendMessage", [
            'chat_id' => $chat_id,
            'text' => "🔗 Please send the post link for: {$service['name']}"
        ]);
        exit;
    }

    if ($data == "cancel_order") {
        $user['step'] = null;
        $user['order_temp'] = [];
        saveJson(USER_FILE, $users);
        apiRequest("editMessageText", [
            'chat_id' => $chat_id,
            'message_id' => $callback['message']['message_id'],
            'text' => "❌ Order cancelled."
        ]);
        exit;
    }

    if ($data == "confirm_order") {
        $temp = $user['order_temp'];
        $service = getService($temp['service_key']);
        $resp = placeOrder($service['service_id'], $temp['link'], $temp['quantity']);

        if (!isset($resp['order'])) {
            apiRequest("editMessageText", [
                'chat_id' => $chat_id,
                'message_id' => $callback['message']['message_id'],
                'text' => "❌ Failed to place order. Please try again."
            ]);
            $user['step'] = null;
            $user['order_temp'] = [];
            saveJson(USER_FILE, $users);
            exit;
        }

        $order_id = $resp['order'];
        $orders = loadJson(ORDER_FILE);
        $orders[] = [
            'user_id' => $chat_id,
            'service' => $service['name'],
            'link' => $temp['link'],
            'quantity' => $temp['quantity'],
            'price' => $temp['price'],
            'api_order_id' => $order_id,
            'time' => time()
        ];
        saveJson(ORDER_FILE, $orders);

        apiRequest("editMessageText", [
            'chat_id' => $chat_id,
            'message_id' => $callback['message']['message_id'],
            'text' => "✅ Order placed successfully!\n🆔 API Order ID: $order_id"
        ]);

        $user['step'] = null;
        $user['order_temp'] = [];
        saveJson(USER_FILE, $users);
    }
}
